import os
import re
import whisper
import yt_dlp
from flask import Flask, request, render_template_string

app = Flask(__name__)
model = whisper.load_model("base")  # Use "small", "medium", or "large" for better accuracy

HTML_TEMPLATE = '''
<!doctype html>
<title>YouTube Audio Transcriber</title>
<h1>Convert YouTube Video to Transcript (No Captions Required)</h1>
<form method="post">
  YouTube URL: <input type="text" name="url" required>
  <input type="submit" value="Transcribe">
</form>
{% if transcript %}
<h2>Transcript</h2>
<pre>{{ transcript }}</pre>
{% elif error %}
<p style="color: red;">Error: {{ error }}</p>
{% endif %}
'''

def extract_video_id(url):
    match = re.search(r'(?:v=|youtu\.be/)([\w-]{11})', url)
    return match.group(1) if match else None

def download_audio(video_url, video_id):
    output_filename = f"{video_id}.mp3"
    ydl_opts = {
        'format': 'bestaudio/best',
        'outtmpl': f'{video_id}.%(ext)s',
        'quiet': True,
        'postprocessors': [{
            'key': 'FFmpegExtractAudio',
            'preferredcodec': 'mp3',
            'preferredquality': '192',
        }],
        #'ffmpeg_location': 'ffmpeg',  # Use full path if needed
        'ffmpeg_location': r'C:\ffmpeg\bin',  # <-- Hardcoded path here
    }

    with yt_dlp.YoutubeDL(ydl_opts) as ydl:
        ydl.download([video_url])

    if not os.path.exists(output_filename):
        raise FileNotFoundError(f"Expected audio file {output_filename} not found.")
    
    return output_filename

@app.route('/', methods=['GET', 'POST'])
def index():
    transcript = ""
    error = ""
    if request.method == 'POST':
        url = request.form['url']
        video_id = extract_video_id(url)
        if not video_id:
            error = "Invalid YouTube URL"
        else:
            try:
                audio_file = download_audio(url, video_id)
                result = model.transcribe(audio_file)
                transcript = result['text']
                os.remove(audio_file)
            except Exception as e:
                error = f"Failed: {e}"
    return render_template_string(HTML_TEMPLATE, transcript=transcript, error=error)

if __name__ == '__main__':
    app.run(debug=True)
